<?php
/* --------------------------------------------------------------
   TextPhraseRepository.php 2020-04-23
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Language\Repositories;

use Gambio\Core\Cache\FileCache;
use Gambio\Core\Language\LanguageId;
use Psr\SimpleCache\CacheInterface;
use Psr\SimpleCache\InvalidArgumentException;

/**
 * Class TextPhraseRepository
 *
 * @package Gambio\Core\Language\Repositories
 */
class TextPhraseRepository implements \Gambio\Core\Language\TextPhraseRepository
{
    /**
     * @var FileCache
     */
    private $textCache;
    
    /**
     * @var TextPhraseReader
     */
    private $reader;
    
    
    /**
     * TextPhraseRepository constructor.
     *
     * @param CacheInterface   $textCache
     * @param TextPhraseReader $reader
     */
    public function __construct(CacheInterface $textCache, TextPhraseReader $reader)
    {
        $this->textCache = $textCache;
        $this->reader    = $reader;
    }
    
    
    /**
     * @param string     $section
     * @param LanguageId $languageId
     *
     * @return string[]
     */
    public function getSectionPhrases(string $section, LanguageId $languageId): array
    {
        $cacheKey = $section . '_lang_' . $languageId->value();
        try {
            if ($this->textCache->has($cacheKey)) {
                return $this->textCache->get($cacheKey, []);
            }
        } catch (InvalidArgumentException $e) {
            return [];
        }
        
        $textPhrases = $this->reader->getSectionPhrases($section, $languageId);
        try {
            $this->textCache->set($cacheKey, $textPhrases);
        } catch (InvalidArgumentException $e) {
        }
        
        return $textPhrases;
    }
    
    
    /**
     * @param string     $phrase
     * @param string     $section
     * @param LanguageId $languageId
     *
     * @return string
     */
    public function getPhraseText(string $phrase, string $section, LanguageId $languageId): string
    {
        $textPhrases = $this->getSectionPhrases($section, $languageId);
        
        return array_key_exists($phrase, $textPhrases) ? $textPhrases[$phrase] : $phrase;
    }
}